#include <string.h>

#include "textcodec.h"
#include "iso2.h"
#include "win1250.h"

#ifdef UNIX
#define stricmp strcasecmp
#endif

uchar16_t u16latin[] =
{
    0x00,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //0, 0x0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //8, 0x8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //16, 0x10
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //24, 0x18
    32,33,34,35,2,37,38,39,                     //32, 0x20
    40,41,42,43,44,45,46,47,                    //40, 0x28
    48,49,50,51,52,53,54,55,                    //48, 0x30
    56,57,58,59,60,61,62,63,                    //56, 0x38
    64,65,66,67,68,69,70,71,                    //64, 0x40
    72,73,74,75,76,77,78,79,                    //72, 0x48
    80,81,82,83,84,85,86,87,                    //80, 0x50
    88,89,90,91,92,93,94,95,                    //88, 0x58
    96,97,98,99,100,101,102,103,                //96, 0x60
    104,105,106,107,108,109,110,111,            //104, 0x68
    112,113,114,115,116,117,118,119,            //112, 0x70
    120,121,122,123,124,125,126,0,              //120, 0x78
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //128, 0x80
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //136, 0x88
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //144, 0x90
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //152, 0x98
    0xbd,142,0xbd,1,36,3,0xbd,141,              //160, 0xa0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,    //168, 0xa8
    137,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,     //176, 0xb0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,144,     //184, 0xb8
    'A','A','A','A','A','A','A','C',            //192, 0xc0
    'E','E','E','E','I','I','I','I',            //200, 0xc8
    'D','N','O','O','O','O','O',0xbd,           //208, 0xd0
    //'O','U','U','U','U','Y',222,'ss',            //216, 0xd8 the last are Þ and ß
    'O','U','U','U','U','Y','T',223,            //216, 0xd8 the last are Þ (cap Thorn) and ß
    'a','a','a','a','a','a','a','c',            //224, 0xe0
    'e','e','e','e','i','i','i','i',            //232, 0xe8
    'd','n','o','o','o','o','o',0xbd,           //240, 0xf0
    'o','u','u','u','u','y','t','y',            //248, 0xf8 (thorn)
    'A','a','A','a','A','a','C','c',            //256, 0x100
    'C','c','C','c','C','c','D','d',            //264, 0x108
    'D','d','E','e','E','e','E','e',            //272, 0x110
    'E','e','E','e','G','g','G','g',            //280, 0x118
    'G','g','G','g','H','h','H','h',            //288, 0x120
    'I','i','I','i','I','i','I','i',            //296, 0x128
    'I','i',0xbd,0xbd,'J','j','K','k',          //304, 0x130
    'k','L','l','L','l','L','l','L',            //312, 0x138
    'l','L','l','N','n','N','n','N',            //320, 0x140
    'n','n','N','n','O','o','O','o',            //328, 0x148
    'O','o',0xbd,0xbd,'R','r','R','r',          //336, 0x150
    'R','r','S','s','S','s','S','s',            //344, 0x158
    'S','s','T','t','T','t','T','t',            //352, 0x160
    'U','u','U','u','U','u','U','u',            //360, 0x168
    'U','u','U','u','W','w','Y','y',            //368, 0x170
    'Y','Z','z','Z','z','Z','z',0xbd,           //376, 0x178
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //384, 0x180
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //392, 0x188
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //400, 0x190
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //408, 0x198
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //416, 0x1a0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //424, 0x1a8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //432, 0x1b0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //440, 0x1b8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //448, 0x1c0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //456, 0x1c8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //464, 0x1d0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //472, 0x1d8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //480, 0x1e0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //488, 0x1e8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //496, 0x1f0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //504, 0x1f8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //512, 0x200
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //520, 0x208
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //528, 0x210
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //536, 0x218
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //544, 0x220
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //552, 0x228
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //560, 0x230
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //568, 0x238
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //576, 0x240
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //584, 0x248
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //592, 0x250
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //600, 0x258
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //608, 0x260
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //616, 0x268
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //624, 0x270
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //632, 0x278
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //640, 0x280
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //648, 0x288
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //656, 0x290
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //664, 0x298
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //672, 0x2a0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //680, 0x2a8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //688, 0x2b0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //696, 0x2b8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //704, 0x2c0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //712, 0x2c8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //720, 0x2d0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //728, 0x2d8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //736, 0x2e0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //744, 0x2e8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //752, 0x2f0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //760, 0x2f8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //768, 0x300
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //776, 0x308
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //784, 0x310
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //792, 0x318
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //800, 0x320
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //808, 0x328
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //816, 0x330
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //824, 0x338
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //832, 0x340
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //840, 0x348
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //848, 0x350
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //856, 0x358
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //864, 0x360
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //872, 0x368
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //880, 0x370 -- start Greek
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //888, 0x378
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,902,0xbd,      //896, 0x380  
    904,905,906,0xbd,908,0xbd,910,911,      //904, 0x388
    912,913,914,915,916,917,918,919,      //912, 0x390
    920,921,922,923,924,925,926,927,      //920, 0x398
    928,929,962,931,932,933,934,935,      //928, 0x3a0
    936,937,938,939,902,904,905,906,      //936, 0x3a8
    944,913,914,915,916,917,918,919,      //944, 0x3b0
    920,921,922,923,924,925,926,927,      //952, 0x3b8
    928,929,962,931,932,933,934,935,      //960, 0x3c0
    936,937,938,939,908,910,911,0xbd,      //968, 0x3c8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //976, 0x3d0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //984, 0x3d8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //992, 0x3e0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //1000, 0x3e8
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //1008, 0x3f0
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,      //1016, 0x3f8 -- end Greek
    0xbd,1025,1026,1027,1028,1029,1030,1031,      //1024, 0x400 -- start Cyrillic
    1032,1033,1034,1035,1036,0xbd,1038,1039,      //1032, 0x408
    1040,1041,1042,1043,1044,1045,1046,1047,      //1040, 0x410
    1048,1049,1050,1051,1052,1053,1054,1055,      //1048, 0x418
    1056,1057,1058,1059,1060,1061,1062,1063,      //1056, 0x420
    1064,1065,1066,1067,1068,1069,1070,1071,      //1064, 0x428
    1040,1041,1042,1043,1044,1045,1046,1047,      //1072, 0x430
    1048,1049,1050,1051,1052,1053,1054,1055,      //1080, 0x438
    1056,1057,1058,1059,1060,1061,1062,1063,      //1088, 0x440
    1064,1065,1066,1067,1068,1069,1070,1071,      //1096, 0x448
    0xbd,1025,1026,1027,1028,1029,1030,1031,      //1104, 0x450
    1032,1033,1034,1035,1036,0xbd,1038,1039,      //1112, 0x458
    0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd      //1120, 0x460
};

const uint32_t u16max = sizeof(u16latin)/sizeof(uchar16_t);


const char* cpNames[] = {"utf-8", "iso-8859-2", "windows-1250"};

TextCodec::eEncoding TextCodec::srcEncoding = TextCodec::Utf8;
TextCodec::eEncoding TextCodec::destEncoding = TextCodec::Utf8;
unsigned char TextCodec::m_u16map[0x10000];
unsigned short* TextCodec::m_pgmap = NULL;

void TextCodec::selectSourceEncoding(const char* name)
{
    m_pgmap = NULL;

    if (stricmp(name, cpNames[0]) == 0) {
        srcEncoding = Utf8;
    } else if (stricmp(name, cpNames[1]) == 0) {
        srcEncoding = Iso2;
        m_pgmap = iso2;
    } else if (stricmp(name, cpNames[2]) == 0) {
        srcEncoding = Win1250;
        m_pgmap = win1250;
    }
}

void TextCodec::selectTargetEncoding(const char* name)
{
    unsigned short* pgmap = NULL;

    if (stricmp(name, cpNames[0]) == 0) {
        destEncoding = Utf8;
    } else if (stricmp(name, cpNames[1]) == 0) {
        destEncoding = Iso2;
        pgmap = iso2;
    } else if (stricmp(name, cpNames[2]) == 0) {
        destEncoding = Win1250;
        pgmap = win1250;
    }

    if (pgmap)
	for (int i = 0; i < 256; i++)
    	    m_u16map[pgmap[i]] = i;
}

unsigned short TextCodec::utf8To16(unsigned char** p)
{
    unsigned short c16, d;
    unsigned short trailing;
    unsigned char* pc = *p;

    d = *(pc++);
    if (d < 0x80) {
        c16 = d;
        trailing = 0;
    } else if (d < 0xC0)
        return 0xFFFF;    // trailing byte in leading position
    else if (d < 0xE0) {
        c16 = d & 0x1F;
        trailing = 1;
    } else if (d < 0xF0) {
        c16 = d & 0x0F;
        trailing = 2;
    } else if (d < 0xF8) {
        c16 = d & 0x07;
        trailing = 3;
    } else
        return 0xFFFF;    // no chance for this in UTF-16
    
    for ( ; trailing; trailing--) {
        if (((d = *(pc++)) & 0xC0) != 0x80)
            return 0xFFFF;
        c16 <<= 6;
        c16 |= d & 0x3F;
    }
    
    *p = pc;

    return c16;
}

int TextCodec::utf16To8(unsigned ucode, char* uc)
{
    int l = 0;
    if (ucode < 0x80) {
        uc[l++] = ucode;
    } else if (ucode < 0x800) {
        uc[l++] = (ucode >> 6) + 0xC0;
        uc[l++] = (ucode & 63) + 0x80;
    } else if (ucode < 0x10000) {
        uc[l++] = (ucode >> 12) + 0xE0;
        uc[l++] = ((ucode >> 6) & 63) + 0x80;
        uc[l++] = (ucode & 63) + 0x80;
    } else if (ucode < 2097152) {
        uc[0] = (ucode >> 18) + 0xF0;
        uc[1] = ((ucode >> 12) & 63) + 0x80;
        uc[2] = (ucode >> 6) & 63 + 0x80;
        uc[3] = (ucode & 63) + 0x80;
        l = 4;
    }
    return l;
}

void TextCodec::xcode(const char* strin, char* strout)
{
    char* o = strout;
    unsigned char* p = (unsigned char*)strin;
    unsigned short u16;
    while (*p) {
        u16 = (m_pgmap) ? m_pgmap[*(p++)] : utf8To16(&p);
        if (destEncoding != Utf8) {
            *o = m_u16map[u16];
            o++;
        } else {
            char uc[5];
            char* pu = uc;
            int l = utf16To8(u16, uc);
            while (l) {
                *o = *(pu++);
                o++;
                l--;
            }
        }
    }
    *o = 0;
}

void TextCodec::toUtf16(const char* strin, unsigned short* strout)
{
    unsigned short* o = strout;
    unsigned char* p = (unsigned char*)strin;
    unsigned short u16;
    while (*p) {
        u16 = (m_pgmap) ? m_pgmap[*(p++)] : utf8To16(&p);
        *o = u16;
        o++;
    }
    *o = 0;
}

void TextCodec::toLatin(const char* strin, char* strout)
{
    char* o = strout;
    unsigned char* p = (unsigned char*)strin;
    uchar16_t u16;
    while (*p) {
        u16 = (m_pgmap) ? m_pgmap[*(p++)] : utf8To16(&p);
        u16 = u16latin[u16];
    	if (u16 > 255)
    	{
    	    *o = u16 >> 8 & 0xFF;
    	    o++;
    	    *o = u16 & 0xFF;
    	}
    	else
    	    *o = u16 & 0xFF;
        
        o++;
    }
    *o = 0;
}

void TextCodec::toLatin(const uchar16_t* strin, char* strout)
{
    char* o = strout;
    uchar16_t* p = (uchar16_t*)strin;
    uchar16_t u16;
    while (*p) {
        u16 = *p++;;
        u16 = u16latin[u16];
    	if (u16 > 255)
    	{
    	    *o = u16 >> 8 & 0xFF;
    	    o++;
    	    *o = u16 & 0xFF;
    	}
    	else
    	    *o = u16 & 0xFF;
        
        o++;
    }
    *o = 0;
}
