//--------------------------------------------------------------------------------------
//  Plik DwaWejscia.cpp
//  Filtr pozwala na mieszanie ze sob obrazw z dwch rnych rde.
//	Obrazy powinny mie te same rozmiary.
//  Autor: Piotr M. Szczypinski
//  2009.10.15
//  Instytut Elektroniki Politechniki Ldzkiej
//--------------------------------------------------------------------------------------


#include <streams.h>

#include <windows.h>
#include <initguid.h>
#include <olectl.h>

#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include "dwawejscia.h"

#pragma warning(disable:4238)  


//------------------------------------------------------------------------------
// Struktury opisu filtru potrzebne w procesie rejestracji filtru w systemie.

const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
	&MEDIATYPE_Video,   
	&MEDIASUBTYPE_NULL  
};

const AMOVIESETUP_PIN psudPins[] =
{
	{
		L"Input (L)", 
			FALSE,       
			FALSE,          
			FALSE,        
			FALSE,        
			&CLSID_NULL,       
			L"Output",       
			1,               
			&sudPinTypes },   
		{
			L"Input (R)",   
				FALSE,           
				FALSE,           
				FALSE,          
				FALSE,        
				&CLSID_NULL,    
				L"Output",     
				1,           
				&sudPinTypes },  
			{ L"Output",      
			FALSE,         
			TRUE,            
			FALSE,          
			FALSE,           
			&CLSID_NULL,   
			L"Input",           
			1,                 
			&sudPinTypes    
			}
};

const AMOVIESETUP_FILTER sudContrast =
{
	&CLSID_VDwaWej,       
	L"KL_DWAWEJSCIA",  
	MERIT_DO_NOT_USE,     
	3,                 
	psudPins        
};


CFactoryTemplate g_Templates[1] = 
{
	{ L"KL_DWAWEJSCIA"
	, &CLSID_VDwaWej
	, CFiltrDwaWejscia::CreateInstance
	, NULL
	, &sudContrast }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);

//==============================================================================
// KLASA: CVTwoInpLIn

//------------------------------------------------------------------------------
CVTwoInpLIn::CVTwoInpLIn(
						 TCHAR *pObjectName,
						 CTransformFilter *pTransformFilter,
						 HRESULT * phr,
						 LPCWSTR pName):
CTransformInputPin(pObjectName, pTransformFilter, phr, pName)
{
}

//==============================================================================
// KLASA: CVTwoInpRIn

//------------------------------------------------------------------------------
CVTwoInpRIn::CVTwoInpRIn(
						 TCHAR *pObjectName,
						 CTransformFilter *pTransformFilter,
						 HRESULT * phr,
						 LPCWSTR pName):
CTransformInputPin(pObjectName, pTransformFilter, phr, pName)
{
}

//------------------------------------------------------------------------------
HRESULT
CVTwoInpRIn::Receive(IMediaSample * pSample)
{
	HRESULT hr;
	ASSERT(pSample);
	hr = CBaseInputPin::Receive(pSample);
	if (S_OK == hr) 
	{
		hr = ((CFiltrDwaWejscia*)m_pTransformFilter)->ReceiveR(pSample);
	}
	return hr;
}


//==============================================================================
// KLASA: CFiltrDwaWejscia

//------------------------------------------------------------------------------
HRESULT
CFiltrDwaWejscia::ReceiveR(IMediaSample *pSample)
{
	AM_SAMPLE2_PROPERTIES * const pProps = m_pInputR->SampleProps();
	if (pProps->dwStreamId != AM_STREAM_MEDIA) 
	{
		return S_OK;
	}
	HRESULT hr;
	ASSERT(pSample);
	ASSERT (m_pOutput != NULL) ;
	hr = TransformR(pSample);
	return hr;
}

//------------------------------------------------------------------------------
CFiltrDwaWejscia::CFiltrDwaWejscia(TCHAR *tszName,LPUNKNOWN punk,HRESULT *phr) :
CTransformFilter(tszName, punk, CLSID_VDwaWej),
m_pInputR(NULL),
m_bFlushinghere(false) 
{
	HRESULT hr;
	ASSERT(tszName);
	ASSERT(phr);
	ghLeftEvent = CreateEvent(NULL, TRUE, FALSE, TEXT("ReleaseRight"));
	ghRightEvent = CreateEvent(NULL, TRUE, FALSE, TEXT("RightIn"));
	m_pInput = new CVTwoInpLIn(NAME("Input (L)"), this, &hr, L"Input (L)"); 
	m_pInputR = new CVTwoInpRIn(NAME("Input (R)"), this, &hr, L"Input (R)"); 
	m_pOutput = new CTransformOutputPin(NAME("Output"),this,  &hr, L"Output");
}

//------------------------------------------------------------------------------
CFiltrDwaWejscia::~CFiltrDwaWejscia()
{
	CloseHandle(ghLeftEvent);
	CloseHandle(ghRightEvent);
	delete m_pInputR;
}

//------------------------------------------------------------------------------
// CreateInstance jest wywolywana przez GraphBuilder w celu utworzenia filtra
CUnknown * WINAPI CFiltrDwaWejscia::CreateInstance(LPUNKNOWN punk, HRESULT *phr) 
{
	ASSERT(phr);

	CFiltrDwaWejscia *pNewObject = new CFiltrDwaWejscia(NAME("KL_Dwa_Wejscia"), punk, phr);
	if (pNewObject == NULL) 
	{
		if (phr)
			*phr = E_OUTOFMEMORY;
	}
	return pNewObject;
}


//------------------------------------------------------------------------------
// Funkcja odblokowuje watki lewego i prawego obrazu. 
// Przekazuje komende BeginFlush do nastepnego filtra.
HRESULT
CFiltrDwaWejscia::BeginFlush(void)
{
	HRESULT hr = NOERROR;
	if (m_pOutput != NULL) 
	{
		m_bFlushinghere = true;
		SetEvent(ghRightEvent);
		SetEvent(ghLeftEvent);
		hr = m_pOutput->DeliverBeginFlush();
	}
	return hr;
}

//------------------------------------------------------------------------------
// Przekazuje komende EndFlush do nastepnego filtra.
HRESULT
CFiltrDwaWejscia::EndFlush(void)
{
	m_bFlushinghere = false;
	ASSERT (m_pOutput != NULL);
	return m_pOutput->DeliverEndFlush();
}

//------------------------------------------------------------------------------
HRESULT CFiltrDwaWejscia::StartStreaming()
{
	m_bFlushinghere = false;
	return NOERROR;
}
HRESULT CFiltrDwaWejscia::StopStreaming()
{
	m_bFlushinghere = true;
	SetEvent(ghRightEvent);
	SetEvent(ghLeftEvent);
	return NOERROR;
}




//------------------------------------------------------------------------------
// Transform odbiera obraz z wejscia lewego. 
// Odpowiada za synchronizacje czasowa z prawym obrazem	
HRESULT CFiltrDwaWejscia::Transform(IMediaSample *pIn, IMediaSample *pOut)
{
	REFERENCE_TIME bt, et;
	REFERENCE_TIME btr, etr;
	HRESULT hr;

	if(m_bFlushinghere)
	{
		return S_FALSE;
	}
	if(pIn == NULL)
	{
		return E_POINTER;
	}
	if(WaitForSingleObject(ghRightEvent, 1000)==WAIT_TIMEOUT)
	{
		m_bFlushinghere = true;
		SetEvent(ghRightEvent);
		SetEvent(ghLeftEvent);
		pRightSample = NULL;
		return S_FALSE;
	}
	if(m_bFlushinghere)
	{
		return S_FALSE;
	}
	if(pRightSample == NULL) 
	{
		return S_FALSE;
	}
	hr = pIn->GetTime(&bt, &et);
	if(hr == S_OK || hr == VFW_S_NO_STOP_TIME)
	{
		for(int k = 0; k < 2; k++)
		{
			hr = pRightSample->GetTime(&btr, &etr);
			if(hr == S_OK)
			{
				if((btr + etr)/2 < bt)  
				{
					ResetEvent(ghRightEvent);
					SetEvent(ghLeftEvent);
					if(WaitForSingleObject(ghRightEvent, 1000)==WAIT_TIMEOUT)
					{
						m_bFlushinghere = true;
						SetEvent(ghRightEvent);
						SetEvent(ghLeftEvent);
						pRightSample = NULL;
						return S_FALSE;
					}
					if(m_bFlushinghere) 
					{
						return S_FALSE;
					}
				}
				else break;
			}
			if(hr == VFW_S_NO_STOP_TIME)
			{
				if(btr + 250 < bt)
				{
					ResetEvent(ghRightEvent);
					SetEvent(ghLeftEvent);
					if(WaitForSingleObject(ghRightEvent, 1000)==WAIT_TIMEOUT)
					{
						m_bFlushinghere = true;
						SetEvent(ghRightEvent);
						SetEvent(ghLeftEvent);
						pRightSample = NULL;
						return S_FALSE;
					}

					if(m_bFlushinghere) 
					{
						return S_FALSE;
					}
				}
				else break;
			}
		}
	}
	if(pRightSample == NULL) 
	{
		return S_FALSE; 
	}

	hr =  InitializeOutSample(pIn, pOut);
	if (FAILED(hr)) 
	{
		return hr;
	}
	MergeImages(pIn, pRightSample, pOut);
	return NOERROR;
}



//------------------------------------------------------------------------------
// TransformR odbiera obraz z wejscia prawego. 
// Pozwala na wstrzymanie watku do momentu nadejscia obrazu prawego.	
HRESULT CFiltrDwaWejscia::TransformR(IMediaSample *pIn)
{
	if(m_bFlushinghere)
	{
		pRightSample = NULL;
		return S_FALSE;
	}
	pRightSample = pIn;
	if(	SignalObjectAndWait(ghRightEvent, ghLeftEvent, 1000, FALSE)==WAIT_TIMEOUT)
	{
		m_bFlushinghere = true;
		SetEvent(ghRightEvent);
		SetEvent(ghLeftEvent);
		pRightSample = NULL;
		return S_FALSE;
	}
	if(m_bFlushinghere)
	{
		pRightSample = NULL;
		return S_FALSE;
	}
	ResetEvent(ghRightEvent);
	ResetEvent(ghLeftEvent);
	pRightSample = NULL;
	return NOERROR;
}

//------------------------------------------------------------------------------
HRESULT CFiltrDwaWejscia::InitializeOutSample(IMediaSample *pSource, IMediaSample *pDest) const
{
	CheckPointer(pSource,E_POINTER);
	CheckPointer(pDest,E_POINTER);
	long lSourceSize = pSource->GetActualDataLength();
	REFERENCE_TIME TimeStart, TimeEnd;
	if(NOERROR == pSource->GetTime(&TimeStart, &TimeEnd))
	{
		pDest->SetTime(&TimeStart, &TimeEnd);
	}
	LONGLONG MediaStart, MediaEnd;
	if(pSource->GetMediaTime(&MediaStart,&MediaEnd) == NOERROR)
	{
		pDest->SetMediaTime(&MediaStart,&MediaEnd);
	}
	HRESULT hr = pSource->IsSyncPoint();
	if(hr == S_OK)
	{
		pDest->SetSyncPoint(TRUE);
	}
	else if(hr == S_FALSE)
	{
		pDest->SetSyncPoint(FALSE);
	}
	else
	{  
		return E_UNEXPECTED;
	}
	AM_MEDIA_TYPE *pMediaType;
	pSource->GetMediaType(&pMediaType);
	pDest->SetMediaType(pMediaType);
	DeleteMediaType(pMediaType);
	hr = pSource->IsPreroll();
	if(hr == S_OK)
	{
		pDest->SetPreroll(TRUE);
	}
	else if(hr == S_FALSE)
	{
		pDest->SetPreroll(FALSE);
	}
	else
	{
		return E_UNEXPECTED;
	}
	hr = pSource->IsDiscontinuity();
	if(hr == S_OK)
	{
		pDest->SetDiscontinuity(TRUE);
	}
	else if(hr == S_FALSE)
	{
		pDest->SetDiscontinuity(FALSE);
	}
	else
	{
		return E_UNEXPECTED;
	}
	pDest->SetActualDataLength(lSourceSize);
	return NOERROR;
}

//------------------------------------------------------------------------------
// Glowna funkcja przetwarzania. 
// Otrzymuje lewy (pLIn) i prawy (pRIn) obraz, 
// zwraca obraz wynikowy w obiekcie wskazanym przez pOut. 
HRESULT CFiltrDwaWejscia::MergeImages(IMediaSample *pLIn, IMediaSample *pRIn, IMediaSample *pOut) const
{
	CheckPointer(pLIn, E_POINTER);
	CheckPointer(pRIn, E_POINTER);
	CheckPointer(pOut, E_POINTER);

	BYTE *pLInBuffer, *pRInBuffer, *pOutBuffer;

	long lOutSize = pOut->GetActualDataLength();
	long lLInSize = pLIn->GetActualDataLength();
	long lRInSize = pRIn->GetActualDataLength();

	pLIn->GetPointer(&pLInBuffer);
	pRIn->GetPointer(&pRInBuffer);
	pOut->GetPointer(&pOutBuffer);
	int stride = (m_Width * sizeof( RGBTRIPLE ) + 3) & -4;

	// @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	// @@@@   TU UZUPELNIC KOD O ALGORYTM OBLICZANIA DYSPARYCJI      @@@@@@@@@@@@@@@
	// @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	// pLInBuffer, pRInBuffer oraz pOutBuffer sa wskaznikami do poczatkow blokow
	// obrazow. Obrazy sa w formacie 24 bity na piksel. Liczbe bitow na
	// piksel okresla zmienna m_Bits=24. Uklad bitow okresla struktur RGBTRIPLE. 
	// Wymiary obrazu w pikselach podaja zmienne m_Height i m_Width.
	// Wysokosc obrazu wyjsciowego moze byc ustawiona na podwojna wysokosc
	// obrazow wejsciowych (patrz komentarz w metodzie GetMediaType).

	// Ten fragment usrednia obraz:
	int x, y;
	RGBTRIPLE *prgbl;      
	RGBTRIPLE *prgbr;      
	RGBTRIPLE *prgbo;      
	for (y = 0; y < m_Height; y++)
	{
		for (x = 0; x < m_Width; x++) 
		{
			prgbl = (RGBTRIPLE*) (pLInBuffer + stride * y + sizeof(RGBTRIPLE)*x);
			prgbr = (RGBTRIPLE*) (pRInBuffer + stride * y + sizeof(RGBTRIPLE)*x);
			prgbo = (RGBTRIPLE*) (pOutBuffer + stride * y + sizeof(RGBTRIPLE)*x);

			prgbo->rgbtGreen = (prgbl->rgbtGreen + prgbr->rgbtGreen)/2;
			prgbo->rgbtBlue = (prgbl->rgbtBlue + prgbr->rgbtBlue)/2;
			prgbo->rgbtRed = (prgbl->rgbtRed + prgbr->rgbtRed)/2;
		}
	}

	// @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	// @@@@                          KONIEC                                     @@@@
	// @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

	return NOERROR;

}

//------------------------------------------------------------------------------
// Metoda informujaca o akceptowanych formatach danych wejsciowych i wyjsciowych
HRESULT CFiltrDwaWejscia::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	if(iPosition < 0)
	{
		return E_INVALIDARG;
	}
	if(iPosition > 0)
	{
		return VFW_S_NO_MORE_ITEMS;
	}

	CheckPointer(pMediaType,E_POINTER);

	VIDEOINFO *pvi = (VIDEOINFO *) pMediaType->AllocFormatBuffer(sizeof(VIDEOINFO));
	if(NULL == pvi)
		return(E_OUTOFMEMORY);

	ZeroMemory(pvi, sizeof(VIDEOINFO));

	pvi->bmiHeader.biCompression = BI_RGB;
	pvi->bmiHeader.biBitCount = 24;
	HDC hdc = GetDC(NULL);  
	//    UINT res = GetSystemPaletteEntries(hdc, 0, iPALETTE_COLORS, (LPPALETTEENTRY) &(pvi->TrueColorInfo.bmiColors));
	ReleaseDC(NULL, hdc);

	pvi->bmiHeader.biSize       = sizeof(BITMAPINFOHEADER);
	// Aby wysokosc lub szerokosc obrazu wyjsciowego byla inna od wejsciowych 
	// nalezy zmodyfikowac dwie ponizsze linie. Np. podstawienie 
	//    pvi->bmiHeader.biHeight = m_Height * 2;
	// Spowoduje, ze wysokosc obrazu wyjsciowego bedzie 2x wieksza od
	// wejsciowego.
	pvi->bmiHeader.biWidth      = m_Width;
	pvi->bmiHeader.biHeight     = m_Height;

	pvi->bmiHeader.biPlanes     = 1;
	pvi->bmiHeader.biSizeImage  = GetBitmapSize(&pvi->bmiHeader);
	pvi->bmiHeader.biClrImportant = 0;

	SetRectEmpty(&(pvi->rcSource));
	SetRectEmpty(&(pvi->rcTarget)); 
	pMediaType->SetType(&MEDIATYPE_Video);
	pMediaType->SetFormatType(&FORMAT_VideoInfo);
	pMediaType->SetTemporalCompression(FALSE);
	const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
	pMediaType->SetSubtype(&SubTypeGUID);
	pMediaType->SetSampleSize(pvi->bmiHeader.biSizeImage);
	return NOERROR;
} 

//------------------------------------------------------------------------------
// Metoda sprawdzajaca format danych wejsciowych
HRESULT CFiltrDwaWejscia::CheckInputType(const CMediaType *mtIn)
{
	VIDEOINFO *pvi;
	CheckPointer(mtIn,E_POINTER);
	if(*mtIn->FormatType() != FORMAT_VideoInfo)
	{
		return E_INVALIDARG;
	}
	if((IsEqualGUID(*mtIn->Type(), MEDIATYPE_Video)) &&
		(IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24)))
	{
		pvi = (VIDEOINFO *) mtIn->Format();
		if(pvi->bmiHeader.biBitCount == 24) 
		{	 
			m_Height = pvi->bmiHeader.biHeight;                 
			m_Width = pvi->bmiHeader.biWidth;   
			m_Bits = pvi->bmiHeader.biBitCount;  
			return NOERROR;
		}
	}
	return E_FAIL;
}

//------------------------------------------------------------------------------
// Metoda sprawdzajaca format danych wejsciowych i wyjsciowych
HRESULT CFiltrDwaWejscia::CheckTransform(const CMediaType *mtIn,const CMediaType *mtOut)
{
	CheckPointer(mtIn,E_POINTER);
	CheckPointer(mtOut,E_POINTER);
	HRESULT hr;
	if(FAILED(hr = CheckInputType(mtIn)))
	{
		return hr;
	}
	if(*mtOut->FormatType() != FORMAT_VideoInfo)
	{
		return E_INVALIDARG;
	}
	if(mtIn->FormatLength() < sizeof(VIDEOINFOHEADER) ||
		mtOut->FormatLength() < sizeof(VIDEOINFOHEADER))
		return E_INVALIDARG;
	VIDEOINFO *pInput  = (VIDEOINFO *) mtIn->Format();
	VIDEOINFO *pOutput = (VIDEOINFO *) mtOut->Format();

	if(pOutput->bmiHeader.biCompression != BI_RGB) return E_INVALIDARG;
	if(pOutput->bmiHeader.biBitCount != 24) return E_INVALIDARG;
	if(pInput->bmiHeader.biCompression != BI_RGB) return E_INVALIDARG;
	if(pInput->bmiHeader.biBitCount != 24) return E_INVALIDARG;
	if(pOutput->bmiHeader.biWidth != pInput->bmiHeader.biWidth) return E_INVALIDARG; 
	if(pOutput->bmiHeader.biHeight != pInput->bmiHeader.biHeight) return E_INVALIDARG;
	return NOERROR;

}

//------------------------------------------------------------------------------
HRESULT CFiltrDwaWejscia::DecideBufferSize(IMemAllocator *pAlloc,ALLOCATOR_PROPERTIES *pProperties)
{
	CheckPointer(pAlloc,E_POINTER);
	CheckPointer(pProperties,E_POINTER);

	HRESULT hr = NOERROR;

	VIDEOINFO *pvi = (VIDEOINFO *) m_pOutput->CurrentMediaType().Format();
	pProperties->cBuffers = 1;
	pProperties->cbBuffer = pvi->bmiHeader.biSizeImage;

	ASSERT(pProperties->cbBuffer);

	ALLOCATOR_PROPERTIES Actual;
	hr = pAlloc->SetProperties(pProperties,&Actual);
	if(FAILED(hr))
	{
		return hr;
	}
	if(Actual.cbBuffer < pProperties->cbBuffer)
	{
		return E_FAIL;
	}
	ASSERT(Actual.cBuffers == 1);
	return NOERROR;

}

//------------------------------------------------------------------------------
// Metody informujace o liczbie, wskaznikach i nazwach obiektow wyprowadzen
int CFiltrDwaWejscia::GetPinCount()
{
	return 3;
}

CBasePin* CFiltrDwaWejscia::GetPin(int n)
{
	switch(n)
	{
	case 0: return m_pInput;
	case 1: return m_pInputR;
	case 2: return m_pOutput;
	default: return NULL;
	}
}

STDMETHODIMP CFiltrDwaWejscia::FindPin(LPCWSTR Id, IPin **ppPin)
{
	CheckPointer(ppPin,E_POINTER);
	ValidateReadWritePtr(ppPin,sizeof(IPin *));

	if (0==lstrcmpW(Id,L"InL")) 
	{
		*ppPin = GetPin(0);
	}
	else if (0==lstrcmpW(Id,L"InR")) 
	{
		*ppPin = GetPin(1);
	}
	else if (0==lstrcmpW(Id,L"Out")) 
	{
		*ppPin = GetPin(2);
	} else {
		*ppPin = NULL;
		return VFW_E_NOT_FOUND;
	}
	HRESULT hr = NOERROR;
	if (*ppPin) {
		(*ppPin)->AddRef();
	} else {
		hr = E_OUTOFMEMORY; 
	}
	return hr;
}

//------------------------------------------------------------------------------
// Metody wywolywane w procesie rejestrowania lub wyrejestrowania filtru
STDAPI DllRegisterServer()
{
	return AMovieDllRegisterServer2(TRUE);
} 

STDAPI DllUnregisterServer()
{
	return AMovieDllRegisterServer2(FALSE);
} 

//------------------------------------------------------------------------------
// Metody wywolywane w procesie dynamicznej konsolidacji biblioteki dll/ax
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);

BOOL APIENTRY DllMain(HANDLE hModule, 
					  DWORD  dwReason, 
					  LPVOID lpReserved)
{
	return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}

